/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.widgets;

import intrinsic.flash.display.DisplayObjectContainer;
import intrinsic.mx.containers.*;

import org.eclipse.swt.*;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;

public class TabFolder extends Composite {
	TabItem [] items;
	int selection = -1;

public TabFolder (Composite parent, int style) {
	super (parent, checkStyle (style));
}

void addChild(Widget child, int index) {
}

void addItem (TabItem item) {
	if (items == null) items = new TabItem [4];
	int length = items.length;
	for (int i=0; i<length; i++) {
		if (items [i] == item) return;
	}
	int index = 0;
	while (index < length) {
		if (items [index] == null) break;
		index++;
	}
	if (index == length) {
		TabItem [] temp = new TabItem [length + 4];
		System.arraycopy (items, 0, temp, 0, length);
		items = temp;
	}
	items [index] = item;
	((TabNavigator)object).addChild(item.object);
}

public void addSelectionListener (SelectionListener listener) {
	checkWidget ();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	TypedListener typedListener = new TypedListener(listener);
	addListener(SWT.Selection,typedListener);
	addListener(SWT.DefaultSelection,typedListener);
}

static int checkStyle (int style) {
	style = checkBits (style, SWT.TOP, SWT.BOTTOM, 0, 0, 0, 0);
	/*
	* Even though it is legal to create this widget
	* with scroll bars, they serve no useful purpose
	* because they do not automatically scroll the
	* widget's client area.  The fix is to clear
	* the SWT style.
	*/
	return style & ~(SWT.H_SCROLL | SWT.V_SCROLL);
}

protected void checkSubclass () {
	if (!isValidSubclass ()) error (SWT.ERROR_INVALID_SUBCLASS);
}

void createWidget() {
	super.createWidget();
	items = new TabItem [4];
}

void createHandle() {
	TabNavigator widget = new TabNavigator();
	if (widget == null) error(SWT.ERROR_NO_HANDLES);
	object = widget;
}

void createItem (TabItem item, int index) {
	int itemCount = getItemCount();
	if (!(0 <= index && index <= itemCount)) error (SWT.ERROR_INVALID_RANGE);
	item.createWidget ();
	
	//TODO INDEX
	addItem(item);
//	if (itemCount == count) error (SWT.ERROR_ITEM_NOT_ADDED);
}

Image _getImage(DisplayObjectContainer item) {
	return items [item.parent.getChildIndex(item)].image;
}

public TabItem getItem (int index) {
	checkWidget ();
	int count = getItemCount ();
	if (!(0 <= index && index < count)) error (SWT.ERROR_INVALID_RANGE);
	return items [index];
}

public int getItemCount () {
	checkWidget ();
	int count = 0;
	while (count < items.length && items[count] != null) count++;
	return count;
}

public TabItem[] getItems () {
	checkWidget ();
	int count = getItemCount ();
	TabItem [] result = new TabItem [count];
	System.arraycopy (items, 0, result, 0, count);
	return result;
}

public TabItem[] getSelection () {
	checkWidget ();
	int index = getSelectionIndex ();
	if (index == -1) return new TabItem [0];
	return new TabItem [] {items [index]};
}

public int getSelectionIndex () {
	checkWidget ();
	return selection;
}

public int indexOf (TabItem item) {
	checkWidget ();
	if (item == null) error (SWT.ERROR_NULL_ARGUMENT);
	for (int i=0; i<items.length; i++) {
		if (items [i] == item) return i;
	}
	return -1;
}

Point minimumSize (int wHint, int hHint, boolean flushCache) {
	Control [] children = _getChildren ();
	int width = 0, height = 0;
	for (int i=0; i<children.length; i++) {
		Control child = children [i];
		int index = 0;	
		int count = items.length;
		while (index < count) {
			if (items [index].control == child) break;
			index++;
		}
		if (index == count) {
			Rectangle rect = child.getBounds ();
			width = Math.max (width, rect.x + rect.width);
			height = Math.max (height, rect.y + rect.height);
		} else {
			Point size = child.computeSize (wHint, hHint, flushCache);
			width = Math.max (width, size.x);
			height = Math.max (height, size.y);
		}
	}
	return new Point (width, height);
}

void releaseChildren (boolean destroy) {
	if (items != null) {
		int count = getItemCount ();
		for (int i=0; i<count; i++) {
			TabItem item = items [i];
			if (item != null && !item.isDisposed ()) {
				item.release (false);
			}
		}
		items = null;
	}
	super.releaseChildren (destroy);
}

void removeControl (Control control) {
	super.removeControl (control);
	for (int i=0; i<items.length; i++) {
		TabItem item = items [i];
		if (item != null && item.control == control) {
			item.setControl (null);
		}
	}
}

void removeChild(Widget child) {
}

void removeItem (TabItem item) {
	int index = 0;
	while (index < items.length) {
		if (items [index] == item) break;
		index++;
	}
	System.arraycopy (items, index + 1, items, index, items.length - index);
	items [items.length - 1] = null;
}

public void removeSelectionListener (SelectionListener listener) {
	checkWidget ();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (eventTable == null) return;
	eventTable.unhook (SWT.Selection, listener);
	eventTable.unhook (SWT.DefaultSelection,listener);	
}

void sendSelection (int selection) {
	setSelection (selection, true);
}

public void setSelection (int index) {
	checkWidget ();
	int count = getItemCount ();
	if (!(0 <= index && index < count)) return;
	setSelection (index, false);
}

void setSelection (int index, boolean notify) {
	int oldIndex = selection;
	if (oldIndex == index) return;
	if (oldIndex != -1) {
		TabItem item = items [oldIndex];
		Control control = item.control;
		if (control != null && !control.isDisposed ()) {
			//control.setVisible (false);
		}
	}
	TabNavigator widget = (TabNavigator)object;
	widget.selectedIndex = index;
	int newIndex = widget.selectedIndex;
	if (newIndex != -1) {
		TabItem item = items [newIndex];
		Control control = item.control;
		if (control != null && !control.isDisposed ()) {
			//control.setBounds (getClientArea ());
			//control.setVisible (true);
		}
		if (notify) {
			Event event = new Event ();
			event.item = item;
			sendEvent (SWT.Selection, event);
		}
	}
}

public void setSelection (TabItem[] items) {
	checkWidget ();
	if (items == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (items.length == 0) {
		setSelection (-1, false);
	} else {
		for (int i=items.length-1; i>=0; --i) {
			int index = indexOf (items [i]);
			if (index != -1) setSelection (index, false);
		}
	}
}

public void setSelection (TabItem item) {
	checkWidget ();
	if (item == null) error (SWT.ERROR_NULL_ARGUMENT);
	setSelection (new TabItem [] {item});
}

}
